
/**
* generates a visualisation of the discrete fourier transform of a sample
*/

import krister.Ess.*;
import java.awt.FileDialog;

void setup() {

  size(500,500);
  colorMode(HSB);

  Ess.start(this);
  AudioChannel sampleInput = new AudioChannel();

  FileDialog fileDialog = new FileDialog(frame, "open", FileDialog.LOAD);
  fileDialog.setVisible(true);
  String url="";
  if (fileDialog.getDirectory()!=null) {
    url=fileDialog.getDirectory() + fileDialog.getFile();
  }

  sampleInput.loadSound(url);
  int dataLength=sampleInput.samples.length;
  float[] sample = new float[dataLength];
  for (int i=0;i<sampleInput.samples.length;i++) {
    sample[i]=sampleInput.samples[i];
  }  

  int wLength = 1000;
  int sampleRate = 44100;

  //discrete fourier transform
  int nbParts=ceil(dataLength/wLength);
  // for crossfaded, put int nbParts=ceil(dataLength/wLength)*2;
  int nbHarm=wLength;
  // for audible part, put int nbHarm=ceil(wLength/2);
  float[][] sinPart = new float[nbHarm][nbParts];
  float[][] cosPart = new float[nbHarm][nbParts];  
  for (int partial=0;partial<nbHarm;partial++) {
    for (int part=0;part<nbParts;part++) {
      sinPart[partial][part]=0;
      cosPart[partial][part]=0;
      for (int i=0;i<wLength;i++) {
        float phasor=(float)(i+part*wLength)*partial*TWO_PI/wLength;
        sinPart[partial][part]+=sample[(i+part*wLength)%dataLength]*sin(phasor);
        cosPart[partial][part]+=sample[(i+part*wLength)%dataLength]*cos(phasor);
      }
    }
  }

  //getting frequency magnitude and phase
  float[][] freq = new float[nbHarm][nbParts];
  float[][] magn = new float[nbHarm][nbParts];
  float[][] phas = new float[nbHarm][nbParts];
  for (int partial=0;partial<nbHarm;partial++) {
    for (int part=0;part<nbParts;part++) {
      freq[partial][part]=(float)partial*sampleRate/(float)wLength;
      magn[partial][part]=2.0*sqrt(sq(sinPart[partial][part])+sq(cosPart[partial][part]))/(float)wLength;
      // value in db is 20.0*log10(magn[partial][part])
      phas[partial][part]=atan2(cosPart[partial][part],sinPart[partial][part]);
    }
  }

  displayDFT(freq,magn,phas);

  float[] resynthesis = resynth(freq,magn,phas,wLength,sampleRate);
  // for crossfades, put float[] resynthesis = resynth2(freq,magn,phas,wLength,sampleRate);

  //displayWaveform(resynthesis);

  AudioChannel sampleOutput = new AudioChannel();
  sampleInput.initChannel(sample.length);
  sampleOutput.initChannel(resynthesis.length);
  sample = normalizeSample(sample);
  resynthesis = normalizeSample(resynthesis);
  for (int i=0;i<sample.length;i++) {
    sampleInput.samples[i]=sample[i];
  }
  for (int i=0;i<resynthesis.length;i++) {
    sampleOutput.samples[i]=resynthesis[i];
  }
  sampleInput.saveSound(dataPath("input.wav"));
  sampleOutput.saveSound(dataPath("output.wav"));
  Ess.stop();
}

float[] normalizeSample(float[] sample) {
  float maxVolume=0;
  for (int i=0;i<sample.length;i++) {
    if (abs(sample[i])>maxVolume) {
      maxVolume=abs(sample[i]);
    }
  }
  for (int i=0;i<sample.length;i++) {
    sample[i]=sample[i]/maxVolume;
  }
  return sample;
}

float[] newSaw(int dataLength, int sampleRate) {
  //synthesize a saw wave;
  float[] sample = new float[dataLength];
  float fSaw=50;
  for (int i=0;i<dataLength;i++) {
    fSaw+=0.1;
    int periodLength = floor((float)sampleRate/fSaw);
    sample[i] = i%periodLength;
    sample[i] = map(sample[i],0,periodLength,-1,1);
  }
  return sample;
}

float[] resynth(float[][] freq,float[][] magn,float[][] phas,int wLength, int sampleRate) {
  int sLength = wLength*freq[0].length;
  int nbHarm = freq.length;
  float[] sample= new float[sLength];
  for (int i=0;i<sLength;i++) {
    sample[i] = 0;
  }
  for (int i=0;i<sLength;i++) {
    int cPart = floor((float)i/(wLength));//which part of the sample are we in
    int cPartS = i%(wLength);//which sample on this part
    float thisSample=0;
    for (int partial=0;partial<nbHarm;partial++) {
      float phasor=(float)cPartS*TWO_PI/sampleRate*freq[partial][cPart]+phas[partial][cPart];
      thisSample+=sin(phasor)*magn[partial][cPart];
    }
    sample[i]=thisSample;
  }
  return sample;
}

float[] resynth2(float[][] freq,float[][] magn,float[][] phas,int wLength, int sampleRate) {
  int sLength = wLength*freq[0].length;
  int nbHarm = freq.length;
  float[] sample= new float[sLength];
  for (int i=0;i<sLength;i++) {
    sample[i] = 0;
  }
  int nbParts=ceil(sLength/wLength)*2;
  float[][] parts = new float[nbParts][sLength/nbParts];
  for (int p=0;p<nbParts;p++) {
    for (int i=0;i<parts[0].length;i++) {      
      parts[p][i]=0;
      for (int partial=0;partial<nbHarm;partial++) {
        float phasor=(float)i*TWO_PI/sampleRate*freq[partial][p]+phas[partial][p];
        parts[p][i]+=sin(phasor)*magn[partial][p];
      }
    }
  }

  for (int i=0;i<sLength;i++) {
    int cPart = floor((float)i/(wLength)*2);//which part of the sample are we in
    int cPartS = i%(wLength/2);//which sample on this part
    sample[i]=parts[cPart][cPartS+wLength]*(1-cPartS/(wLength/2))+parts[(cPart+1)%nbParts][cPartS]*(cPartS/(wLength/2));
  }

  return sample;
}

void displayDFT(float[][] freq,float[][] magn,float[][] phas) {
  background(0);
  int nbHarm = freq.length;
  int nbParts = freq[0].length;
  for (int partial=0;partial<nbHarm;partial++) {
    for (int part=0;part<nbParts;part++) {
      stroke(map(phas[partial][part],0,TWO_PI,0,0xFF),0xE0,max(0xFF-sq(magn[partial][part]/2),0));
      line(part*width/nbParts,height-partial,(part+2)*width/nbParts,height-partial);
    }
  }
}

void displayTransform(float[][] transform) {
  background(0);
  int nbParts=transform[0].length;
  for (int part=0;part<nbParts;part++) {  
    for (int partial=0;partial<transform.length;partial++) {
      stroke(transform[partial][part]);
      line(part*width/nbParts,height-partial,(part+1)*width/nbParts,height-partial);
    }
  }
}

void displayWaveform(float[] sample) {
  background(0);
  stroke(0xFF);
  for (int i=0;i<sample.length-1;i++) {
    float position = map(sample[i],-1,1,0,height);
    float position2 = map(sample[i+1],-1,1,0,height);
    line(i/5,position,(i+1)/5,position2);
  }
}

// Calculates the base-10 logarithm of a number
float log10 (float x) {
  return (log(x) / log(10));
}

